import bpy
import bmesh
from mathutils import *

def create_beam(start, end, width, name):
    """
    Create a rectangular beam between two points
    """
    bm = bmesh.new()

    bmesh.ops.create_cube(bm, size=1)
    bmesh.ops.scale(bm, vec=(width, width, (end - start).length), verts=bm.verts)

    # rotate and translate the beam to the correct location
    rot_diff = (end - start).rotation_difference((0,0,1))
    bmesh.ops.rotate(bm, cent=(0,0,0), matrix=rot_diff.to_matrix(), verts=bm.verts)
    bmesh.ops.translate(bm, vec=start + (end - start) / 2, verts=bm.verts)

    mesh_data = bpy.data.meshes.new(name)
    bm.to_mesh(mesh_data)
    bm.free()

    return bpy.data.objects.new(name, mesh_data)

def main():
    # clear all mesh objects
    bpy.ops.object.select_all(action='DESELECT')
    bpy.ops.object.select_by_type(type='MESH')
    bpy.ops.object.delete()

    size = 1
    half_size = size / 2
    width = 0.05  # width of the beams

    # create beams for each edge of the cube
    corners = [
        (-half_size, -half_size, -half_size),
        (-half_size, -half_size, half_size),
        (-half_size, half_size, -half_size),
        (-half_size, half_size, half_size),
        (half_size, -half_size, -half_size),
        (half_size, -half_size, half_size),
        (half_size, half_size, -half_size),
        (half_size, half_size, half_size),
    ]
    edges = [
        (0, 1), (0, 2), (0, 4),
        (7, 6), (7, 5), (7, 3),
        (1, 3), (1, 5),
        (2, 3), (2, 6),
        (4, 5), (4, 6),
    ]

    for i, (start, end) in enumerate(edges):
        start_v = Vector(corners[start])
        end_v = Vector(corners[end])
        beam = create_beam(start_v, end_v, width, f'Beam{i+1}')
        bpy.context.collection.objects.link(beam)

main()
